#include <apr_pools.h>
#include <apr_hash.h>

#include "svn_error.h"
#include "svn_pools.h"
#include "svn_config.h"
#include "svn_client.h"
#include "svn_cmdline.h"


svn_error_t *
print_entry(apr_hash_t *dirents, apr_pool_t *pool) {
	apr_hash_index_t *hi;
	svn_dirent_t *current;
	void *val;
	svn_revnum_t youngest = 0;
	for(hi = apr_hash_first(pool, dirents); hi; hi = apr_hash_next(hi)) {
		apr_hash_this(hi, NULL, NULL, &val);
		current = (svn_dirent_t *) val;
		if(current->created_rev > youngest) {
			youngest = current->created_rev;
		}
	}
	printf("%d\n",youngest);
	
	return SVN_NO_ERROR;
}

svn_error_t *
the_action(const char* url, svn_client_ctx_t *ctx, apr_pool_t *pool) {
	apr_hash_t *dirents;
	svn_opt_revision_t *rev = apr_pcalloc(pool, sizeof(*rev));
	
	rev->kind = svn_opt_revision_unspecified;
	
	SVN_ERR(svn_client_ls(&dirents, url, rev, 0, ctx, pool));
	SVN_ERR(print_entry(dirents, pool));
	
	return SVN_NO_ERROR;
}

int main(int argc, char **argv) {
	if(argc != 2 || *argv[1] == '-') {
		printf("Usage: %s url\n", argv[0]);
		return EXIT_FAILURE;
	}
	
	const char* url = argv[1];
	svn_error_t* err = SVN_NO_ERROR;

	/* Initialize the app. */
	if (svn_cmdline_init ("svnhead", stderr) != EXIT_SUCCESS)
		return EXIT_FAILURE;

	/* Create our top-level pool.  Use a seperate mutexless allocator,
	 * given this application is single threaded.
	 */
	apr_allocator_t *allocator;
	if (apr_allocator_create (&allocator))
		return EXIT_FAILURE;

	apr_allocator_max_free_set (allocator, SVN_ALLOCATOR_RECOMMENDED_MAX_FREE);

	apr_pool_t *pool = svn_pool_create_ex (NULL, allocator);
	apr_allocator_owner_set (allocator, pool);
	
	svn_client_ctx_t *ctx;

	if(!err)
		err = svn_client_create_context(&ctx, pool);
	
	if(!err)
		err = svn_config_get_config(&(ctx->config), NULL, pool);

	/* Authentication set-up. */
	{
		/* The whole list of registered providers */
		apr_array_header_t *providers = apr_array_make (pool, 10, sizeof (svn_auth_provider_object_t *));
		
		svn_auth_provider_object_t *provider;
		
		/* The main disk-caching auth providers, for both
		   'username/password' creds and 'username' creds.  */
		svn_client_get_simple_provider (&provider, pool);
		APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;
		svn_client_get_username_provider (&provider, pool);
		APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;
		/* The server-cert, client-cert, and client-cert-password providers. */
		svn_client_get_ssl_server_trust_file_provider (&provider, pool);
		APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;
		svn_client_get_ssl_client_cert_file_provider (&provider, pool);
		APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;
		svn_client_get_ssl_client_cert_pw_file_provider (&provider, pool);
		APR_ARRAY_PUSH (providers, svn_auth_provider_object_t *) = provider;

		svn_auth_baton_t *ab;
		svn_auth_open (&ab, providers, pool);
		ctx->auth_baton = ab;
		
		svn_auth_set_parameter(ab, SVN_AUTH_PARAM_NON_INTERACTIVE, "");
		svn_auth_set_parameter(ab, SVN_AUTH_PARAM_DONT_STORE_PASSWORDS, "");
		svn_auth_set_parameter(ab, SVN_AUTH_PARAM_NO_AUTH_CACHE, "");
	}

	if(!err)
		err = the_action(url, ctx, pool);
	
	if(err) {
		printf("Action failed: %s\n", err->message);
		return EXIT_FAILURE;
	} else {
		return EXIT_SUCCESS;
	}
}
